/*
 Rocrail - Model Railroad Software
 Copyright (c) 2002-2018 Robert Jan Versluis, Rocrail.net
 All rights reserved.
*/
#include "mbus.h"
#include "mbus_dfs.h"
#include "cv.h"
#include "io.h"
#include "trace.h"

#include <stdlib.h>
#include <time.h>

//---------- MBus constructor.
MBus::MBus(byte ID, byte BoardType) {
  m_CANID = ID;
  m_UID = m_CANID + UID_ACCESSORY;
  trace("MBus 0x%04X", m_UID );
  m_BoardType = BoardType;
  m_Hash = 0;
  randomSeed(time(NULL));
}

int MBus::getCANBPS() {
  return 250; 
}

byte MBus::getFilters(unsigned long* filters) {
  return 0;
}

void MBus::setID(byte id) {
  m_CANID = id;
  m_UID   = m_CANID + UID_ACCESSORY;
}

bool MBus::evalFrame(iCANFrame frame, iCmdData cmd) {
  bool eval = false;
  cmd->cmd = 0;

  if( frame->id > 0 ) {
    unsigned int l_UID = 0;
    bool rsp = false;
    switch( getCmd(frame, &rsp) ) {
      case CMD_PING:
        if( !rsp ) {
          cmd->cmd = CMD_QUERY;
          eval = true;
        }
      break;
      case CMD_WRITE_CONFIG:
        l_UID = frame->data[2] * 256 + frame->data[3];
        cmd->cmd = CMD_SETCV;
        cmd->addr  = frame->data[4] * 256 + frame->data[5];
        cmd->value = frame->data[6];
        cmd->thisid = (l_UID == m_UID) ? true:false;
        eval = true;
        break;
      case CMD_READ_CONFIG:
        l_UID = frame->data[2] * 256 + frame->data[3];
        cmd->cmd = CMD_GETCV;
        cmd->addr  = frame->data[4] * 256 + frame->data[5];
        cmd->value = frame->data[6];
        cmd->thisid = (l_UID == m_UID) ? true:false;
        eval = true;
        break;
      case CMD_ACC_SWITCH:
        l_UID = frame->data[2] * 256 + frame->data[3];
        if( l_UID - m_UID > 0 && l_UID - m_UID <= MAX_PORTS ) {
          // Short event
          cmd->cmd  = (frame->data[4] > 0 ? CMD_OUTPUT_ON:CMD_OUTPUT_OFF);
          cmd->id   = l_UID;
          cmd->addr = l_UID - m_UID;
          cmd->thisid = true;
          eval = true;
        }
        else if( l_UID == m_UID ) {
          // Long event
          cmd->cmd  = (frame->data[4] > 0 ? CMD_OUTPUT_ON:CMD_OUTPUT_OFF);
          cmd->id   = l_UID;
          cmd->addr = frame->data[7];
          cmd->thisid = true;
          eval = true;
        }
        break;
    }
  }
  return eval;
}

bool MBus::makeFrame(iCANFrame frame, iCmdData cmd) {
  bool didmake = false;
  
  if( cmd->cmd == CMD_GETCV ) {
    frame->data[0] = 0;
    frame->data[1] = 0;
    frame->data[2] = m_UID >> 8;
    frame->data[3] = m_UID & 0xFF;
    frame->data[4] = cmd->addr / 256;
    frame->data[5] = cmd->addr % 256;
    frame->data[6] = cmd->value;
    makeFrame(frame, CMD_READ_CONFIG, 7, frame->data, true);
    didmake = true;
  }
  else if( cmd->cmd == CMD_GETCV ) {
    frame->data[0] = 0;
    frame->data[1] = 0;
    frame->data[2] = m_UID >> 8;
    frame->data[3] = m_UID & 0xFF;
    frame->data[4] = cmd->addr / 256;
    frame->data[5] = cmd->addr % 256;
    frame->data[6] = cmd->value;
    frame->data[7] = 0xC0;
    makeFrame(frame, CMD_WRITE_CONFIG, 8, frame->data, true);
    didmake = true;
  }
  else if( cmd->cmd == CMD_PUBLISH ) {
    frame->data[0] = 0;
    frame->data[1] = 0;
    frame->data[2] = (m_UID >> 8);
    frame->data[3] = (m_UID & 0xFF);
    frame->data[4] = RCAN_VERMAJ;
    frame->data[5] = RCAN_VERMIN;
    frame->data[6] = MANU_ROCRAIL;
    frame->data[7] = m_BoardType;
    makeFrame(frame, CMD_PING, 8, frame->data, true);
    didmake = true;
  }
  else if( cmd->cmd == EVT_OUTPUT ) {
    frame->data[0] = 0;
    frame->data[1] = 0;
    if( CV::get(EE_CONFIG) & CFG_SHORTEVENTS ) {
      frame->data[2] = (m_UID + cmd->addr) >> 8;
      frame->data[3] = (m_UID + cmd->addr) & 0xFF;
      frame->data[4] = cmd->value;
      frame->data[5] = 1;
      makeFrame(frame, CMD_ACC_SWITCH, 6, frame->data, true);
    }
    else {
      frame->data[2] = m_UID  >> 8;
      frame->data[3] = m_UID & 0xFF;
      frame->data[4] = cmd->value;
      frame->data[5] = 1;
      frame->data[6] = cmd->addr >> 8;
      frame->data[7] = cmd->addr & 0xFF;
      makeFrame(frame, CMD_ACC_SWITCH, 8, frame->data, true);
    }
    didmake = true;
  }
  else if( cmd->cmd == EVT_SENSOR ) {
    frame->data[0] = m_CANID >> 8;
    frame->data[1] = m_CANID & 0xFF;
    frame->data[2] = cmd->addr >> 8;
    frame->data[3] = cmd->addr & 0xFF;
    frame->data[4] = (cmd->value > 0 ? 0:1);
    frame->data[5] = cmd->value;
    frame->data[6] = 0; // TimeH
    frame->data[7] = 0; // TimeL
    makeFrame(frame, SENSOR_EVENT, 8, frame->data, true);
    didmake = true;
  }
  return didmake;
}


unsigned int MBus::getCmd(iCANFrame frame, bool* rsp) {
  unsigned int cmd = (frame->id & 0x01FF0000) >> 17;
  *rsp = (frame->id & 0x00010000) ? true:false;
  return cmd; // remove response flag
}


void MBus::makeFrame(iCANFrame frame, unsigned int cmd, byte len, byte* data, bool rsp ) {
  if( m_Hash == 0 )
    genHash();

  cmd <<= 1;
  cmd |= (rsp?MBUS_RESPONSE:0);
  frame->id = cmd;
  frame->id <<= 16;
  frame->id += m_Hash;
  frame->ext = 1;
  frame->dlc = len;
  for( byte i = 0; i < len; i++ )
    frame->data[i] = data[i];
}


void MBus::genHash() {
  m_Hash  = random(8192);
  m_Hash &= 0x1F7F;
  m_Hash |= 0x0300;
}
