 /*
 Rocrail - Model Railroad Software
 Copyright (c) 2002-2018 Robert Jan Versluis, Rocrail.net
 All rights reserved.
*/
#include <mcp_can.h>
#include "RCAN_dfs.h"
#include "board.h"
#include "gpio.h"
#include "trace.h"

// The CAN object:
static MCP_CAN CAN0(10); // Set CS to pin 10

// The target board:
static Board* ms_Board = null;


//---------- Setup the Arduino.
void setup() {
  // Init terminal
  Serial.begin(500000);

  if( CV::get(0) == 0xFF ) {
    CV::resetAll();
  }

  INT8U bps = CAN_125KBPS;
  INT8U mhz = MCP_8MHZ;

  byte l_BoardConfig = CV::get(EE_BOARD);

  if( (l_BoardConfig & CLOCK_MASK) == CLOCK_16MHZ )
    mhz = MCP_16MHZ;
  
  // Select the wanted target board:
  if( (l_BoardConfig & BOARD_MASK) == BOARD_GPIO ) {
    if( (l_BoardConfig & PROTOCOL_MASK) == PROTOCOL_MBUS )
      ms_Board = new GPIO(PROTOCOL_MBUS, (l_BoardConfig&PINLAYOUT_1?1:0));
    else if( (l_BoardConfig & PROTOCOL_MASK) == PROTOCOL_ZBUS )
      ms_Board = new GPIO(PROTOCOL_ZBUS, (l_BoardConfig&PINLAYOUT_1?1:0));
    else
      ms_Board = new GPIO(PROTOCOL_CBUS, (l_BoardConfig&PINLAYOUT_1?1:0));
  }
  
  if(ms_Board != NULL) {
    if( ms_Board->getCANBPS() == 250 )
      bps = CAN_250KBPS;
    else
      bps = CAN_125KBPS;
  }

  trace("CAN bps=%d, MCP2515 %dMHz", ms_Board->getCANBPS(), (mhz == MCP_16MHZ ?16:8));

  // Initialize MCP2515 running at 8MHz with a baudrate of bps and the masks and filters disabled.
  INT8U rc = CAN_OK;
  if( (rc = CAN0.begin(MCP_STDEXT, bps, mhz)) == CAN_OK ) {
    // Check for filters...
    if( ms_Board != null ) {
      unsigned long filters[MAX_FILTERS+1];
      byte nr = ms_Board->getFilters(filters);
      if( nr > 0 ) {
        // filters[0] is mask
        trace("mask=%08lX", filters[0]);
        rc  = CAN0.init_Mask(0, 1, filters[0]); // Init first mask...
        rc |= CAN0.init_Mask(1, 1, filters[0]); // Init first mask...
        if( rc == CAN_OK ) {
          for( byte i = 0; i < nr && i < MAX_FILTERS; i++ ) {
            trace("filters[%d]=%08lX", i, filters[i+1]);
            rc  = CAN0.init_Filt(i,1,filters[i+1]);
            rc |= CAN0.init_Filt(i+2,1,filters[i+1]);
            if( rc != CAN_OK )
              trace("init filter rc=%d", rc);
          }
        }
        else {
          trace("init mask rc=%d", rc);
        }
      }
    }
    // Set operation mode to normal so the MCP2515 sends acks to received data.
    CAN0.setMode(MCP_NORMAL);
  }
  else {
    trace("Error %d MCP2515", rc);
  }
  
}


//---------- The main loop which scans for CAN and console messages.
void loop() {
  CANFrame frame;

  frame.id   = 0;
  frame.dlc  = 0;

  if( CAN0.checkReceive() == CAN_MSGAVAIL ) {
    INT8U rc = CAN0.readMsgBuf(&frame.id, &frame.ext, &frame.dlc, frame.data);
    if( rc == CAN_OK ) {
      sendASCIIFrame(&frame);
    }    
  }
  else {
    if( checkConsole(&frame) ) {
      byte rc = CAN0.sendMsgBuf( frame.id, frame.ext, frame.dlc, frame.data);
      if(rc == CAN_OK) {
        trace("CAN msg send id=%08lX ext=%d dlc=%d %02X%02X%02X%02X%02X%02X%02X%02X", frame.id, frame.ext, frame.dlc, frame.data[0], frame.data[1], frame.data[2], frame.data[3], frame.data[4], frame.data[5], frame.data[6], frame.data[7]);
      }
    }
  }

  if( ms_Board != null ) {
    if( ms_Board->Process(&frame) ) {
      byte rc = CAN0.sendMsgBuf( frame.id, frame.ext, frame.dlc, frame.data);
      if(rc != CAN_OK) {
        trace("Error %d send", rc);
      }
      sendASCIIFrame(&frame);
    }
  }
  
}


//---------- Convert two ASCII chars to a binary byte value.
byte HEXA2Byte( const byte* s ) {
  byte val = 0;
  if( s[0] >= 'A' && s[0] <= 'F' ) val += (s[0] - 'A') + 10;
  else if( s[0] >= 'a' && s[0] <= 'f' ) val += (s[0] - 'a') + 10;
  else val += (s[0] - '0');
  val <<= 4;
  if( s[1] >= 'A' && s[1] <= 'F' ) val += (s[1] - 'A') + 10;
  else if( s[1] >= 'a' && s[1] <= 'f' ) val += (s[1] - 'a') + 10;
  else val += (s[1] - '0');
  return val;
}


//---------- Convert ASCII frame
void ASCIIFrame2CANFrame(char* consoleCmd, iCANFrame frame) {
  frame->id  = 0;
  frame->dlc = 0;
  frame->ext = 0;
  memset( frame->data, 0, 8 );

  // Standard frame :SC020N71001801;
  if( consoleCmd[1] == 'S' ) {
    byte idh = HEXA2Byte(consoleCmd + 2);
    byte idl = HEXA2Byte(consoleCmd + 4);
    frame->id = (idl >> 5 ) + ((idh&0x1F) << 3);
    frame->ext = 0;
    if( consoleCmd[6] == 'N' ) {
      for( byte i = 0; i < 8; i++ ) {
        if( consoleCmd[7+i*2] == ';' || consoleCmd[7+i*2] == '\0' || consoleCmd[7+i*2] == '\n' || consoleCmd[7+i*2] == '\r' )
          break; 
        frame->data[i] = HEXA2Byte(consoleCmd + 7+i*2);
        frame->dlc++;
      }
    }
  }
  
  // Extended frame
  else if(consoleCmd[1] == 'X') {
    unsigned long idh = HEXA2Byte(consoleCmd + 2);
    unsigned long idl = HEXA2Byte(consoleCmd + 4);
    unsigned long edh = HEXA2Byte(consoleCmd + 6);
    unsigned long edl = HEXA2Byte(consoleCmd + 8);
    frame->id += idh << 24;
    frame->id += idl << 16;
    frame->id += edh << 8;
    frame->id += edl;
    frame->ext = 1;
    if( consoleCmd[10] == 'N' ) {
      for( byte i = 0; i < 8; i++ ) {
        if( consoleCmd[11+i*2] == ';' || consoleCmd[11+i*2] == '\0' || consoleCmd[11+i*2] == '\n' || consoleCmd[11+i*2] == '\r' )
          break; 
        frame->data[i] = HEXA2Byte(consoleCmd + 11+i*2);
        frame->dlc++;
      }
    }
  }
  else {
    trace("unsupported [%s]", consoleCmd);
  }
}


//---------- Check the serial monitor for commands
bool checkConsole(iCANFrame frame) {
static char consoleCmd[32];
static byte consoleIdx = 0;
  if( Serial.available() && consoleIdx < 32 ) {
    consoleCmd[consoleIdx] = (char)Serial.read();
    if( consoleCmd[consoleIdx] == '\n' || consoleCmd[consoleIdx] == '\r' ) {
      consoleCmd[consoleIdx] = '\0';
      consoleIdx = 0;

      if( consoleCmd[0] == ':' ) {
        ASCIIFrame2CANFrame(consoleCmd, frame);
        return true;  
      }
      else if( consoleCmd[0] == '=' ) {
        if( ms_Board != null )
          ms_Board->setID( atoi(consoleCmd+1) );
      }
      else if( consoleCmd[0] == '#' ) {
        if( ms_Board != null ) {
          byte port = consoleCmd[1] - '0';
          if( consoleCmd[1] > 'a' )
            consoleCmd[1] - 'a' + 10;
          ms_Board->setPort( port, (byte)(consoleCmd[2]-'0') );
        }
      }
      else if( strcmp( "dump", consoleCmd ) == 0 ) {
        dumpEEPROM(8);
      }
      else if( strcmp( "sod", consoleCmd ) == 0 ) {
        trace("SoD");
        if( ms_Board != null )
          ms_Board->SoD();
      }
      else if( strcmp( "stat", consoleCmd ) == 0 ) {
        if( ms_Board != null )
          ms_Board->dumpPortState();
      }
      else if( strcmp( "init", consoleCmd ) == 0 ) {
        if( ms_Board != null )
          ms_Board->init();
      }
      else if( strcmp( "pub", consoleCmd ) == 0 ) {
        if( ms_Board != null )
          ms_Board->publish();
      }
      else if( strcmp( "loop", consoleCmd ) == 0 ) {
        trace("Mode: Loopback");
        CAN0.setMode(MCP_LOOPBACK);
      }
      else if( strcmp( "norm", consoleCmd ) == 0 ) {
        trace("Mode: Normal");
        CAN0.setMode(MCP_NORMAL);
      }
      else if( strcmp( "mbus", consoleCmd ) == 0 ) {
        byte l_BoardConfig = CV::get(EE_BOARD);
        l_BoardConfig &= ~PROTOCOL_MASK;
        l_BoardConfig |= PROTOCOL_MBUS;
        CV::set(EE_BOARD, l_BoardConfig);
        ResetArduino();
      }
      else if( strcmp( "cbus", consoleCmd ) == 0 ) {
        byte l_BoardConfig = CV::get(EE_BOARD);
        l_BoardConfig &= ~PROTOCOL_MASK;
        l_BoardConfig |= PROTOCOL_CBUS;
        CV::set(EE_BOARD, l_BoardConfig);
        ResetArduino();
      }
      else if( strcmp( "zbus", consoleCmd ) == 0 ) {
        byte l_BoardConfig = CV::get(EE_BOARD);
        l_BoardConfig &= ~PROTOCOL_MASK;
        l_BoardConfig |= PROTOCOL_ZBUS;
        CV::set(EE_BOARD, l_BoardConfig);
        ResetArduino();
      }
      else if( strcmp( "16m", consoleCmd ) == 0 ) {
        byte l_BoardConfig = CV::get(EE_BOARD);
        l_BoardConfig &= ~CLOCK_MASK;
        l_BoardConfig |= CLOCK_16MHZ;
        CV::set(EE_BOARD, l_BoardConfig);
        ResetArduino();
      }
      else if( strcmp( "8m", consoleCmd ) == 0 ) {
        byte l_BoardConfig = CV::get(EE_BOARD);
        l_BoardConfig &= ~CLOCK_MASK;
        l_BoardConfig |= CLOCK_8MHZ;
        CV::set(EE_BOARD, l_BoardConfig);
        ResetArduino();
      }
      else if( strcmp( "l0", consoleCmd ) == 0 ) {
        byte l_BoardConfig = CV::get(EE_BOARD);
        l_BoardConfig &= ~PINLAYOUT_MASK;
        l_BoardConfig |= PINLAYOUT_0;
        CV::set(EE_BOARD, l_BoardConfig);
        if( ms_Board != null )
          ms_Board->setPinLayout(0);
      }
      else if( strcmp( "l1", consoleCmd ) == 0 ) {
        byte l_BoardConfig = CV::get(EE_BOARD);
        l_BoardConfig &= ~PINLAYOUT_MASK;
        l_BoardConfig |= PINLAYOUT_1;
        CV::set(EE_BOARD, l_BoardConfig);
        if( ms_Board != null )
          ms_Board->setPinLayout(1);
      }
    }
    else {
      consoleIdx++;
      consoleCmd[consoleIdx] = '\0';
    }
  }
  return false;
}

void ResetArduino() {
  asm volatile ("  jmp 0");  
}
