/*
 Rocrail - Model Railroad Software
 Copyright (c) 2002-2018 Robert Jan Versluis, Rocrail.net
 All rights reserved.
*/
#include "io.h"
#include "trace.h"
#include "cv.h"
#include <Arduino.h>

//---------- IO constructor.
IO::IO(byte pinlayout) {
#ifdef __AVR_ATmega328P__
  trace("__AVR_ATmega328P__");
#endif
#ifdef __AVR_ATmega328PB__
  trace("__AVR_ATmega328PB__");
#endif
  pinMode(LED_BUILTIN,OUTPUT);

  setPinLayout(pinlayout, false);
}


void IO::setPinLayout(byte pinlayout, bool init) {
  trace("pinlayout=%d", pinlayout);
  if( init ) {
    delay(1000);
    initPorts();
  }
  if( pinlayout == 1 ) {
    // ArduCAN-I/O {A7, A6, A2, A1, A4, A5, A0, 2, A3, 3, 4, 5, 6, 7, 8, 9}
    m_Pins[ 0] = A7;
    m_Pins[ 1] = A6;
    m_Pins[ 2] = A2;
    m_Pins[ 3] = A1;
    m_Pins[ 4] = A4;
    m_Pins[ 5] = A5;
    m_Pins[ 6] = A0;
    m_Pins[ 7] = 2;
    m_Pins[ 8] = A3;
    m_Pins[ 9] = 3;
    m_Pins[10] = 4;
    m_Pins[11] = 5;
    m_Pins[12] = 6;
    m_Pins[13] = 7;
    m_Pins[14] = 8;
    m_Pins[15] = 9;
  }
  else {
    // CANGC2a {2, 3, 4, 5, 6, 7, 8, 9, A7, A6, A5, A4, A3, A2, A1, A0}
    m_Pins[ 0] = 2;
    m_Pins[ 1] = 3;
    m_Pins[ 2] = 4;
    m_Pins[ 3] = 5;
    m_Pins[ 4] = 6;
    m_Pins[ 5] = 7;
    m_Pins[ 6] = 8;
    m_Pins[ 7] = 9;
    m_Pins[ 8] = A7;
    m_Pins[ 9] = A6;
    m_Pins[10] = A5;
    m_Pins[11] = A4;
    m_Pins[12] = A3;
    m_Pins[13] = A2;
    m_Pins[14] = A1;
    m_Pins[15] = A0;
  }
  if( init ) {
    delay(1000);
    initIO(MAX_PORTS);
  }
}



//---------- Init one or all pins.
void IO::initIO(byte portnr) {
  if( portnr < MAX_PORTS ) {
    confIO(portnr, CV::get(EE_PORTCFG+portnr) );
  }
  else {
    for( byte i = 0; i < MAX_PORTS; i++ ) {
      confIO(i, CV::get(EE_PORTCFG+i) );
    }
  }
}


//---------- Init all port as output.
void IO::initPorts() {
  for( byte i = 0; i < MAX_PORTS; i++ ) {
    pinMode(m_Pins[i], OUTPUT);
    m_Port[i].state    = 0;
    m_Port[i].duration = 0;
    m_Port[i].timer    = 0;
  }
}


//---------- Set the optional LED.
void IO::setLED(bool on) {
  // The Nano does not have a LED.
  digitalWrite(LED_BUILTIN,on?HIGH:LOW);
}


//---------- Start of Day.
void IO::SoD() {
  for(byte i = 0; i < MAX_PORTS; i++ )
    m_Port[i].state = SOD;
}


//---------- Set the port duration time for block inputs or pulse outputs.
void IO::setPortDuration(byte port, byte duration) {
  if( port < MAX_PORTS ) {
    m_Port[port].duration = duration;
    m_Port[port].timer = 0;
  }
}


//---------- Reset the port duration and timer.
void IO::resetPortTimer(byte port) {
  if( port < MAX_PORTS ) {
    m_Port[port].duration = 0;
    m_Port[port].timer = 0;
  }
}


//---------- Get the port its current timer value.
byte IO::getPortTimer(byte port) {
  if( port < MAX_PORTS ) {
    return m_Port[port].timer;
  }
  return 0;
}


//---------- Set the port state for saving at power off.
byte IO::setPortState(byte port, byte state) {
  byte statechanged = false;
  if( port < MAX_PORTS ) {
    if( m_Port[port].state != state )
      statechanged = true;
    m_Port[port].state = state;
  }
  return statechanged;
}


//---------- Get the current port state.
byte IO::getPortState(byte port) {
  if( port < MAX_PORTS ) {
    return m_Port[port].state;
  }
  return LOW;
}


//---------- Is the port timer elapsed?
byte IO::isPortTimer(byte port) {
  if( port < MAX_PORTS ) {
    if( m_Port[port].duration > 0 && m_Port[port].timer >= m_Port[port].duration )
      return true;
  }
  return false;
}


//---------- Increase the port timers by one.
void IO::increasePortTimers() {
  for(byte i = 0; i < MAX_PORTS; i++ ) {
    if( m_Port[i].duration > 0 )
      m_Port[i].timer++;
  }
}


//---------- Init a port as output or input.
void IO::confIO(byte port, byte conf) {
  if( (conf & PORTCFG_IO) == PORTCFG_OUT ) {
    trace("port %02d(%02X) set OUTPUT", port, m_Pins[port]);
    pinMode(m_Pins[port], OUTPUT);
  }
  else if( (conf & PORTCFG_IO) == PORTCFG_IN ) {
    trace("port %02d(%02X) set INPUT_PULLUP", port, m_Pins[port]);
    pinMode(m_Pins[port], INPUT_PULLUP);
  }
}


//---------- Set the output port to HIGH or LOW.
void IO::setOutput(byte port, byte on, bool inv) {
  if( port < MAX_PORTS ) {
    if( inv )
      digitalWrite(m_Pins[port], on?LOW:HIGH);
    else
      digitalWrite(m_Pins[port], on?HIGH:LOW);
  }
}


//---------- Get the input port its digital value.
bool IO::getInput(byte port) {
  if( port < MAX_PORTS ) {
    byte val = digitalRead(m_Pins[port]);
    return ((val==LOW) ? true:false);
  }
  return 0;
}
